<?php
# SonicCampGuard
#
# Copyright (c) 2024 Bundeswehr
# Diese Anwendung wurde von Cedric Prietz entwickelt.
# E-Mail: CedricPrietz@bundeswehr.org
# App: 90-8303-54618
#
# Letzte Änderung: 01.04.2025

require_once $_SERVER['DOCUMENT_ROOT'] . '/function.inc.php';
requireLogin();
requireRole(['Bediener', 'Bearbeiter', 'Administrator']);
initializeDatabaseConnection();

header('Content-Type: application/json');

$response = [
    'status' => 'error',
    'icon' => 'error',
    'title' => 'Fehler',
    'message' => '',
    'html' => '',
    'footer' => '' // Footer-Option für den Link hinzufügen
];

if (!isset($_POST['inputAccessCard']) || !isset($_POST['zoneID'])) {
    $response['message'] = 'Karten-ID oder Zonen-ID fehlt.';
    echo json_encode($response);
    exit;
}

$inputAccessCard = $_POST['inputAccessCard'];
$zoneID = $_POST['zoneID'];
$onlyAccompanied = false;

try {

    // 1. Prüfung: Existiert die Karte in der Tabelle 'Identities'?
    $sql = "SELECT * FROM Identities WHERE AccessCard = :inputAccessCard LIMIT 1";
    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':inputAccessCard', $inputAccessCard, PDO::PARAM_STR);
    $stmt->execute();

    $identity = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$identity) {
        $response['message'] = "Diese Karte ist keiner Identität zugeordnet.";
        echo json_encode($response);
        exit;
    }

    // 2. Prüfung: Ist das aktuelle Datum innerhalb des Zeitraums IdentityIN und IdentityOUT?
    $currentDate = date('Y-m-d');
    $identityIn = date('Y-m-d', strtotime($identity['IdentityIN']));
    $identityOut = date('Y-m-d', strtotime($identity['IdentityOUT']));

    if (!($currentDate >= $identityIn && $currentDate <= $identityOut)) {
        $response['message'] = "Kein gültiger Buchungszeitraum.";
        echo json_encode($response);
        exit;
    }

    // 3. Prüfung: Aktualisieren der 'CurrentZone' und 'ZoneEntryTimestamp' in der Tabelle 'Identities'
    if ($identity['CurrentZone'] == $zoneID) {
        // Benutzer befindet sich bereits in der Zone; ausbuchen
        $zoneSql = "SELECT ZoneOnExit FROM Zones WHERE ZoneID = :zoneID";
        $zoneStmt = $pdo->prepare($zoneSql);
        $zoneStmt->bindParam(':zoneID', $zoneID, PDO::PARAM_INT);
        $zoneStmt->execute();

        $zone = $zoneStmt->fetch(PDO::FETCH_ASSOC);

        $newZone = $zone['ZoneOnExit']; // Ausbuchungszone
        $action = "Ausbuchung";
        $actionColor = '#7987a1';

    } else {
        // Benutzer ist nicht in der Zone; einbuchen
        $newZone = $zoneID;
        $action = "Einbuchung";
        $actionColor = '#63a83d';

        // 4. Prüfung: Zonenbeschränkung und Autorisierung nur beim Einbuchen
        $zoneSql = "SELECT Restricted FROM Zones WHERE ZoneID = :zoneID";
        $zoneStmt = $pdo->prepare($zoneSql);
        $zoneStmt->bindParam(':zoneID', $zoneID, PDO::PARAM_INT);
        $zoneStmt->execute();

        $zone = $zoneStmt->fetch(PDO::FETCH_ASSOC);

        if ($zone && $zone['Restricted'] == 1) {
            // Prüfen, ob die Karte für eine eingeschränkte Zone autorisiert ist
            $authSql = "
                    SELECT RestrictedZoneAuthorisation 
                    FROM Positions 
                    WHERE (OwnerAccessCard = :inputAccessCard OR SuccessorAccessCard = :inputAccessCard) 
                    LIMIT 1
                ";
            $authStmt = $pdo->prepare($authSql);
            $authStmt->bindParam(':inputAccessCard', $inputAccessCard, PDO::PARAM_STR);
            $authStmt->execute();

            $authorization = $authStmt->fetch(PDO::FETCH_ASSOC);

            if (!$authorization || $authorization['RestrictedZoneAuthorisation'] == 0) {
                // Benutzer darf garnicht in Sperrzone
                $response['status'] = 'error';
                $response['icon'] = 'error';
                $response['title'] = 'Kein Zutritt';
                $response['message'] = "Die Person hat keine Berechtigung diese Zone zu betreten.";
                echo json_encode($response);
                exit;
            } elseif (!$authorization || $authorization['RestrictedZoneAuthorisation'] == 2) {
                // Benutzer darf mit Begleitung in Sperrzone
                $onlyAccompanied = true;
            }
        }
    }

    // Buchungshistorie Hinzufügen
    addBookingHistory(getZoneDetails($identity['CurrentZone'], 'ZoneName'), getZoneDetails($newZone, 'ZoneName'), $inputAccessCard, $identity['Rank'] . ' ' . $identity['Surname'] . ', ' . $identity['Givenname']);

    // Aktualisierung der Identität mit neuer Zone und Zeitstempel
    $updateSql = "
            UPDATE Identities
            SET CurrentZone = :newZone, ZoneEntryTimestamp = '" . date('Y-m-d H:i:s') . "'
            WHERE AccessCard = :inputAccessCard
        ";
    $updateStmt = $pdo->prepare($updateSql);
    $updateStmt->bindParam(':newZone', $newZone, PDO::PARAM_INT);
    $updateStmt->bindParam(':inputAccessCard', $inputAccessCard, PDO::PARAM_STR);
    $updateStmt->execute();

    // 5. Abfrage der Position-Informationen (Department, PositionName) für die Erfolgsmeldung
    $positionSql = "SELECT Department, PositionName FROM Positions WHERE OwnerAccessCard = :inputAccessCard OR SuccessorAccessCard = :inputAccessCard LIMIT 1";
    $positionStmt = $pdo->prepare($positionSql);
    $positionStmt->bindParam(':inputAccessCard', $inputAccessCard, PDO::PARAM_STR);
    $positionStmt->execute();

    $position = $positionStmt->fetch(PDO::FETCH_ASSOC);

    // Überprüfen, ob 'Notes' in der Identität vorhanden ist, und Footer-Link erstellen
    if ($identity['Notes']) {
        $response['footer'] = "<a href='#'>Notizen</a>";
    }

    // Ergebnis und SweetAlert-Ausgabe basierend auf den Prüfungen
    if ($onlyAccompanied) {
        $response['status'] = 'success';
        $response['icon'] = 'warning';
        $response['title'] = 'Eingeschränkter Zugang';

        // Zusätzliche HTML-Inhalte für die Erfolgsmeldung
        $imagePath = $identity['Image'] ? "/uploads/identities/{$identity['Image']}" : "/assets/images/identityPreviewPlaceholder.png";
        $rankHtml = $identity['Rank'] ? "<p class='mb-2'><strong>Dienstgrad:</strong> {$identity['Rank']}</p>" : ""; // Nur anzeigen, wenn Rank vorhanden ist
        $departmentHtml = $position['Department'] ? "<p class='mb-1'><strong>Dienststelle:</strong> {$position['Department']}</p>" : "";
        $positionNameHtml = $position['PositionName'] ? "<p class='mb-1'><strong>Funktion:</strong> {$position['PositionName']}</p>" : "";

        $response['html'] = "
                <div style='text-align: center;'>
                    <img src='{$imagePath}' alt='Benutzerbild' class='mb-3 w-175px h-200px rounded object-fit-cover'>
                    <p class='mb-2 fw-bold' style='font-size: 36px; color: {$actionColor};'>{$action}</p>
                    <p class='mb-1'><strong>Name:</strong> {$identity['Givenname']} {$identity['Surname']}</p>
                    {$rankHtml}
                    {$positionNameHtml}
                    {$departmentHtml}
                    <p class='text-warning fw-bold'>Der Zugang ist nur in Begleitung erlaubt!</p>
                </div>
            ";

    } else {
        $response['status'] = 'success';
        $response['icon'] = 'success';
        $response['title'] = '';

        // Zusätzliche HTML-Inhalte für die Erfolgsmeldung
        $imagePath = $identity['Image'] ? "/uploads/identities/{$identity['Image']}" : "/assets/images/identityPreviewPlaceholder.png";
        $rankHtml = $identity['Rank'] ? "<p class='mb-1'><strong>Dienstgrad:</strong> {$identity['Rank']}</p>" : ""; // Nur anzeigen, wenn Rank vorhanden ist
        $departmentHtml = $position['Department'] ? "<p class='mb-1'><strong>Dienststelle:</strong> {$position['Department']}</p>" : "";
        $positionNameHtml = $position['PositionName'] ? "<p class='mb-1'><strong>Funktion:</strong> {$position['PositionName']}</p>" : "";

        $response['html'] = "
                <div style='text-align: center;'>
                    <img src='{$imagePath}' alt='Benutzerbild' class='mb-3 w-175px h-200px rounded object-fit-cover'>
                    <p class='mb-2 fw-bold' style='font-size: 36px; color: {$actionColor};'>{$action}</p>
                    <p class='mb-1'><strong>Name:</strong> {$identity['Givenname']} {$identity['Surname']}</p>
                    {$rankHtml}
                    {$positionNameHtml}
                    {$departmentHtml}
                </div>
            ";
    }

    echo json_encode($response);

} catch (PDOException $e) {
    $response['message'] = 'Datenbankfehler: ' . $e->getMessage();
    echo json_encode($response);
}

?>